#!/usr/bin/env python3
# pip install pybis

from pybis import Openbis
from pathlib import Path
import os, sys, json
from datetime import datetime
import random


# --- CONFIG (env vars) ---
OBIS_URL     = os.getenv("OBIS_URL",   "https://localhost:8443")
USER         = os.getenv("OBIS_USER",  "admin")
PASSWORD     = os.getenv("OBIS_PASS",  "changeit")
SPACE        = os.getenv("OBIS_SPACE",   "MYSPACE")
PROJECT      = os.getenv("OBIS_PROJECT", "MYPROJECT")
COLL_CODE    = os.getenv("OBIS_COLL",    "MYPROJECT_EXP_1")
SAMPLE_CODE = os.getenv("OBIS_MYSAMPLE", f"MYSAMPLE_{random.randint(1, 1000)}")
DATASET_TYPE = os.getenv("OBIS_DS_TYPE","RAW_DATA")
SAMPLE_TYPE  = os.getenv("SAMPLE_TYPE","MY_SAMPLE_TYPE")

# Optional: choose what to print on stdout: json | permId | identifier
OUTPUT = os.getenv("OBIS_OUTPUT", "json").lower()

def log(level, msg):
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    print(f"{now} [{level}] {msg}", file=sys.stderr)

def safe_get(getter, *args, **kwargs):
    try:
        return getter(*args, **kwargs)
    except ValueError:
        return None

# --- CONNECT ---
ob = Openbis(
    OBIS_URL,
    verify_certificates=False,
    allow_http_but_do_not_use_this_in_production_and_only_within_safe_networks=True
)


try:
    ob.login(USER, PASSWORD)
    # print(ob)


    pat = ob.get_or_create_personal_access_token(sessionName="My Session")
    # print(f"Hello")
    log("INFO", f"Token created: {pat}")
    log("INFO", f"Token created: {pat.permId}")
    # print(pat.sessionName)
    # print(pat.validFromDate, "→", pat.validToDate)
    # print(pat.owner)


    # Ensure Space
    space = safe_get(ob.get_space, SPACE)
    if not space:
        space = ob.new_space(code=SPACE); space.save()
        log("INFO", f"Space created: /{SPACE}")
    else:
        log("INFO", f"Space exists: /{SPACE}")

    # Ensure Project
    proj_identifier = f"/{SPACE}/{PROJECT}"
    project = safe_get(ob.get_project, proj_identifier)
    if not project:
        project = ob.new_project(code=PROJECT, space=space); project.save()
        log("INFO", f"Project created: {proj_identifier}")
    else:
        log("INFO", f"Project exists: {proj_identifier}")

    # Ensure Collection
    coll_identifier = f"/{SPACE}/{PROJECT}/{COLL_CODE}"
    collection = safe_get(ob.get_collection, coll_identifier)
    if not collection:
        collection = ob.new_collection(code=COLL_CODE, project=project, type="DEFAULT_EXPERIMENT"); collection.save()
        log("INFO", f"Collection created: {coll_identifier} (permId: {collection.permId})")
    else:
        log("INFO", f"Collection exists: {coll_identifier} (permId: {collection.permId})")

    # Ensure Sample Type
    sample_type = safe_get(ob.get_sample_type, SAMPLE_TYPE)
    if not sample_type:
        sample_type = ob.new_sample_type(
            code=SAMPLE_TYPE,
            generatedCodePrefix="MST",
            autoGeneratedCode=True,
            subcodeUnique=True,
            listable=True,
            showContainer=False,
            showParents=True,
            showParentMetadata=False,
            validationPlugin=None,
            description="This is my custom sample type"
        )
        sample_type.save()
        log("INFO", f"Sample Type created: {SAMPLE_TYPE}")
    else:
        log("INFO", f"Sample Type exists: {SAMPLE_TYPE}")

    # Ensure Sample
    samples = ob.get_samples(code=SAMPLE_CODE, space=SPACE)
    if samples:
        sample = samples[0]
        log("INFO", f"Sample exists: {sample.identifier} (permId: {sample.permId}, type: {sample.type.code})")
    else:
        sample = ob.new_sample(code=SAMPLE_CODE, experiment=collection, type=SAMPLE_TYPE)
        sample.save()
        log("INFO", f"Sample created: {sample.identifier} (permId: {sample.permId}, type: {SAMPLE_TYPE})")

    # --- MACHINE-READABLE OUTPUT ON STDOUT ---
   
   
   
    print(json.dumps({                   
        "space": SPACE,
        "project": PROJECT,
        "collection": COLL_CODE,
        "sample_code": SAMPLE_CODE,
        "sample_type": SAMPLE_TYPE,
        "identifier": sample.identifier,
        "permId": sample.permId,
        "token": pat.permId
    }))

    sys.exit(0)

except Exception as e:
    log("ERROR", f"{type(e).__name__}: {e}")
    sys.exit(1)
finally:
    try:
        ob.logout()
    except Exception:
        pass
